<?php
namespace EdgeCore\CPT\Shortcodes\InteractiveText;

use EdgeCore\Lib;

class InteractiveText implements Lib\ShortcodeInterface
{
    private $base;

    public function __construct() {
        $this->base = 'edgtf_interactive_text';

        add_action('vc_before_init', array($this, 'vcMap'));
    }

    /**
     * Returns base for shortcode
     * @return string
     */
    public function getBase() {
        return $this->base;
    }

    /**
     * Maps shortcode to Visual Composer. Hooked on vc_before_init
     */
    public function vcMap() {
        if (function_exists('vc_map')) {
            vc_map(
                array(
                    'name'                      => esc_html__('Edge Interactive Text', 'edgtf-core'),
                    'base'                      => $this->getBase(),
                    'category'                  => esc_html__('by EDGE', 'edgtf-core'),
                    'icon'                      => 'icon-wpb-interactive-text extended-custom-icon',
                    'allowed_container_element' => 'vc_row',
                    'params'                    => array(
                        array(
                            'type'       => 'dropdown',
                            'heading'    => esc_html__('Direction', 'edgtf-core'),
                            'param_name' => 'slider_direction',
                            'value'      => array(
                                esc_html__('Horizontal', 'edgtf-core') => 'horizontal',
                                esc_html__('Vertical', 'edgtf-core')   => 'vertical',
                            )
                        ),
                        array(
                            'type'       => 'dropdown',
                            'heading'    => esc_html__('Alignment', 'edgtf-core'),
                            'param_name' => 'alignment',
                            'value'      => array(
                                esc_html__('Default', 'edgtf-core') => '',
                                esc_html__('Left', 'edgtf-core')    => 'left',
                                esc_html__('Center', 'edgtf-core')  => 'center',
                                esc_html__('Right', 'edgtf-core')   => 'right',
                            ),
                            'dependency' => array('element' => 'slider_direction', 'value' => array('vertical'))
                        ),
                        array(
                            'type'        => 'textfield',
                            'heading'     => esc_html__('Holder Padding', 'edgtf-core'),
                            'param_name'  => 'padding',
                            'description' => esc_html__('Insert padding in form (2% 0 2% 5%)', 'edgtf-core'),
                            'dependency'  => array('element' => 'slider_direction', 'value' => array('vertical'))
                        ),
                        array(
                            'type'       => 'dropdown',
                            'param_name' => 'title_underscore',
                            'heading'    => esc_html__('Enable Underscore after Title', 'edgtf-core'),
                            'value'      => array_flip(haar_edge_get_yes_no_select_array(false, true)),
                        ),
                        array(
                            'type'       => 'param_group',
                            'heading'    => esc_html__('Interactive Text Items', 'edgtf-core'),
                            'param_name' => 'interactive_text_items',
                            'value'      => '',
                            'params'     => array(
                                array(
                                    'type'        => 'textfield',
                                    'param_name'  => 'title',
                                    'heading'     => esc_html__('Title', 'edgtf-core'),
                                    'admin_label' => true
                                ),
                                array(
                                    'type'        => 'textfield',
                                    'param_name'  => 'sup_text',
                                    'heading'     => esc_html__('Superscript Text', 'edgtf-core'),
                                    'admin_label' => true
                                ),
                                array(
                                    'type'       => 'colorpicker',
                                    'param_name' => 'text_color',
                                    'heading'    => esc_html__('Title Color', 'edgtf-core')
                                ),
                                array(
                                    'type'        => 'textfield',
                                    'param_name'  => 'subtitle',
                                    'heading'     => esc_html__('Subtitle', 'edgtf-core'),
                                    'admin_label' => true
                                ),
                                array(
                                    'type'       => 'colorpicker',
                                    'param_name' => 'subtitle_color',
                                    'heading'    => esc_html__('Subtitle Color', 'edgtf-core')
                                ),
                                array(
                                    'type'       => 'textfield',
                                    'param_name' => 'link',
                                    'heading'    => esc_html__('Link', 'edgtf-core')
                                ),
                                array(
                                    'type'        => 'attach_image',
                                    'param_name'  => 'image',
                                    'heading'     => esc_html__('Background Image', 'edgtf-core'),
                                    'description' => esc_html__('Select image from media library', 'edgtf-core')
                                ),
                            )
                        ),
                    )
                )
            );
        }
    }

    /**
     * Renders shortcodes HTML
     *
     * @param $atts array of shortcode params
     * @param $content string shortcode content
     * @return string
     */
    public function render($atts, $content = null) {
        $args = array(
            'slider_direction'       => 'horizontal',
            'alignment'              => 'center',
            'padding'                => '',
            'title_underscore'       => 'yes',
            'interactive_text_items' => ''
        );

        $params = shortcode_atts($args, $atts);
        $params['content'] = $content;
        $params['interactive_text_items'] = json_decode(urldecode($params['interactive_text_items']), true);
        $params['classes'] = $this->getSliderClasses($params);
        $params['holder_style'] = $this->getHolderStyle($params);

        $html = edgtf_core_get_shortcode_module_template_part('templates/interactive-text', 'interactive-text', '', $params);

        return $html;
    }

    private function getSliderClasses($params) {
        $classes = array();

        if ($params['slider_direction'] !== '') {
            $classes[] = 'edgtf-it-direction-' . $params['slider_direction'];
        } else {
            $classes[] = 'edgtf-it-direction-horizontal';
        }

        if ($params['alignment'] !== '') {
            $classes[] = 'edgtf-it-alignment-' . $params['alignment'];
        } else {
            $classes[] = 'edgtf-it-alignment-center';
        }

        if ($params['title_underscore'] == 'yes') {
            $classes[] = 'edgtf-it-title-underscore';
        }

        return implode(' ', $classes);
    }

    private function getHolderStyle($params) {
        $style = array();

        if ($params['padding'] !== '') {
            $style[] = 'padding: ' . $params['padding'];
        }

        return implode(' ', $style);
    }
}